/*
 * USB serial driver for USB to UART(s) chip.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * System required:
 * Kernel version beyond 3.2.x
 * Update Log:
 * V1.0 - initial version
 * V1.1 - add support for kernel version beyond 6.5.x
 * V1.2 - add chip with PID 0xE203
 */

#define DEBUG
#define VERBOSE_DEBUG

#undef DEBUG
#undef VERBOSE_DEBUG

#include <linux/errno.h>
#include <linux/idr.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/seq_file.h>
#include <linux/serial.h>
#include <linux/slab.h>
#include <linux/tty.h>
#include <linux/tty_driver.h>
#include <linux/tty_flip.h>
#include <linux/uaccess.h>
#include <linux/usb.h>
#include <linux/usb/cdc.h>
#include <linux/version.h>
#include <asm/byteorder.h>
#include <asm/unaligned.h>

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 11, 0))
#include <linux/sched/signal.h>
#endif

#include "ztekser.h"

#define DRIVER_AUTHOR "ZTEKSER"
#define DRIVER_DESC   "USB serial driver"
#define VERSION_DESC  "V1.2 On 2025.07"

#define IOCTL_MAGIC	      'W'
#define IOCTL_CMD_GETCHIPTYPE _IOR(IOCTL_MAGIC, 0x84, u16)
#define IOCTL_CMD_GETUARTINDEX _IOR(IOCTL_MAGIC, 0x85, u16)
#define IOCTL_CMD_CTRLIN      _IOWR(IOCTL_MAGIC, 0x90, u16)
#define IOCTL_CMD_CTRLOUT     _IOW(IOCTL_MAGIC, 0x91, u16)

static struct usb_driver usbser_driver;
static struct tty_driver *usbser_tty_driver;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 5, 0))
static DEFINE_IDR(usbser_minors);
#else
static struct usbser *usbser_table[USBSER_TTY_MINORS];
#endif
static DEFINE_MUTEX(usbser_minors_lock);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(6, 1, 0))
static void usbser_tty_set_termios(struct tty_struct *tty, const struct ktermios *termios_old);
#else
static void usbser_tty_set_termios(struct tty_struct *tty, struct ktermios *termios_old);
#endif

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 5, 0))
/*
 * Look up an usbser structure by minor. If found and not disconnected, increment
 * its refcount and return it with its mutex held.
 */
static struct usbser *usbser_get_by_index(unsigned int minor)
{
	struct usbser *usbser;

	mutex_lock(&usbser_minors_lock);
	usbser = idr_find(&usbser_minors, minor);
	if (usbser) {
		mutex_lock(&usbser->mutex);
		if (usbser->disconnected) {
			mutex_unlock(&usbser->mutex);
			usbser = NULL;
		} else {
			tty_port_get(&usbser->port);
			mutex_unlock(&usbser->mutex);
		}
	}
	mutex_unlock(&usbser_minors_lock);
	return usbser;
}

static int usbser_alloc_minor(struct usbser *usbser)
{
	int minor;

	mutex_lock(&usbser_minors_lock);
	minor = idr_alloc(&usbser_minors, usbser, 0, USBSER_TTY_MINORS, GFP_KERNEL);
	mutex_unlock(&usbser_minors_lock);

	return minor;
}

/* Release the minor number associated with 'usbser'. */
static void usbser_release_minor(struct usbser *usbser)
{
	mutex_lock(&usbser_minors_lock);
	idr_remove(&usbser_minors, usbser->minor);
	mutex_unlock(&usbser_minors_lock);
}

#else

/*
 * Look up an USBSER structure by index. If found and not disconnected, increment
 * its refcount and return it with its mutex held.
 */
static struct usbser *usbser_get_by_index(unsigned int index)
{
	struct usbser *usbser;

	mutex_lock(&usbser_minors_lock);
	usbser = usbser_table[index];
	if (usbser) {
		mutex_lock(&usbser->mutex);
		if (usbser->disconnected) {
			mutex_unlock(&usbser->mutex);
			usbser = NULL;
		} else {
			tty_port_get(&usbser->port);
			mutex_unlock(&usbser->mutex);
		}
	}
	mutex_unlock(&usbser_minors_lock);

	return usbser;
}

/*
 * Try to find an available minor number and if found, associate it with 'usbser'.
 */
static int usbser_alloc_minor(struct usbser *usbser)
{
	int minor;

	mutex_lock(&usbser_minors_lock);
	for (minor = 0; minor < usbser_TTY_MINORS; minor++) {
		if (!usbser_table[minor]) {
			usbser_table[minor] = usbser;
			break;
		}
	}
	mutex_unlock(&usbser_minors_lock);

	return minor;
}

/* Release the minor number associated with 'usbser'. */
static void usbser_release_minor(struct usbser *usbser)
{
	mutex_lock(&usbser_minors_lock);
	usbser_table[usbser->minor] = NULL;
	mutex_unlock(&usbser_minors_lock);
}

#endif
static int usbser_control_out(struct usbser *usbser, u8 request, u16 value, u16 index)
{
	int retval;

	retval = usb_autopm_get_interface(usbser->control);
	if (retval)
		return retval;
	retval = usb_control_msg(usbser->dev, usb_sndctrlpipe(usbser->dev, 0), request,
				 USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_OUT, value, index, NULL, 0,
				 DEFAULT_TIMEOUT);
	usb_autopm_put_interface(usbser->control);

	return retval;
}

static int usbser_control_in(struct usbser *usbser, u8 request, u16 value, u16 index, char *buf, unsigned bufsize)
{
	int retval;

	retval = usb_autopm_get_interface(usbser->control);
	if (retval)
		return retval;
	retval = usb_control_msg(usbser->dev, usb_rcvctrlpipe(usbser->dev, 0), request,
				 USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_IN, value, index, buf, bufsize,
				 DEFAULT_TIMEOUT);
	usb_autopm_put_interface(usbser->control);

	return retval;
}

static int usbser_control_msg_out(struct usbser *usbser, u8 request, u8 requesttype, u16 value, u16 index, void *buf,
				 unsigned bufsize)
{
	int retval;
	char *buffer;

	buffer = kmalloc(bufsize, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	retval = copy_from_user(buffer, (char __user *)buf, bufsize);
	if (retval)
		goto out;

	retval = usb_autopm_get_interface(usbser->control);
	if (retval)
		goto out;
	retval = usb_control_msg(usbser->dev, usb_sndctrlpipe(usbser->dev, 0), request, requesttype, value, index, buf,
				 bufsize, DEFAULT_TIMEOUT);
	usb_autopm_put_interface(usbser->control);

out:
	kfree(buffer);
	return retval;
}

static int usbser_control_msg_in(struct usbser *usbser, u8 request, u8 requesttype, u16 value, u16 index, void *buf,
				unsigned bufsize)
{
	int retval;
	char *buffer;

	buffer = kmalloc(bufsize, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	retval = usb_autopm_get_interface(usbser->control);
	if (retval)
		goto out;
	retval = usb_control_msg(usbser->dev, usb_rcvctrlpipe(usbser->dev, 0), request, requesttype, value, index, buffer,
				 bufsize, DEFAULT_TIMEOUT);
	if (retval > 0) {
		if (copy_to_user((char __user *)buf, buffer, retval)) {
			retval = -EFAULT;
		}
	}
	usb_autopm_put_interface(usbser->control);

out:
	kfree(buffer);
	return retval;
}

static inline int usbser_set_control(struct usbser *usbser, int control)
{
	if (usbser->iface <= 1)
		return usbser_control_out(usbser, CMD_C2 + usbser->iface, ~control, 0x0000);
	else if (usbser->iface <= 3)
		return usbser_control_out(usbser, CMD_C2 + 0x10 + (usbser->iface - 2), ~control, 0x0000);
	else
		return -1;
}

static inline int usbser_set_line(struct usbser *usbser, struct usb_cdc_line_coding *line)
{
	return 0;
}

static int usbser_get_status(struct usbser *usbser)
{
	char *buffer;
	int retval;
	const unsigned size = 2;
	unsigned long flags;

	buffer = kmalloc(size, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	retval = usbser_control_in(usbser, CMD_R, CMD_C3 + usbser->iface, 0, buffer, size);
	if (retval != size)
		goto out;

	spin_lock_irqsave(&usbser->read_lock, flags);
	usbser->ctrlin = (~(*buffer)) & USBSER_CTI_ST;
	spin_unlock_irqrestore(&usbser->read_lock, flags);

out:
	kfree(buffer);
	return retval;
}

static int usbser_configure(struct usbser *usbser)
{
	char *buffer;
	int r;
	const unsigned size = 8;
	u8 chiptype;
	u8 chipver;

	buffer = kmalloc(size, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	r = usbser_control_in(usbser, CMD_C6, 0, 0, buffer, size);
	if (r <= 0)
		goto out;

	chipver = buffer[0];
	chiptype = buffer[1];
	usbser->iosupport = true;
	usbser->num_ports = 1;
	usbser->chiptype = CHIP_USBSER;

	r = usbser_get_status(usbser);
	if (r < 0)
		goto out;
out:
	kfree(buffer);
	return r < 0 ? r : 0;
}

static int usbser_wb_alloc(struct usbser *usbser)
{
	int i, wbn;
	struct usbser_wb *wb;

	wbn = 0;
	i = 0;
	for (;;) {
		wb = &usbser->wb[wbn];
		if (!wb->use) {
			wb->use = 1;
			return wbn;
		}
		wbn = (wbn + 1) % USBSER_NW;
		if (++i >= USBSER_NW)
			return -1;
	}
}

static int usbser_wb_is_avail(struct usbser *usbser)
{
	int i, n;
	unsigned long flags;

	n = USBSER_NW;
	spin_lock_irqsave(&usbser->write_lock, flags);
	for (i = 0; i < USBSER_NW; i++)
		n -= usbser->wb[i].use;
	spin_unlock_irqrestore(&usbser->write_lock, flags);
	return n;
}

static void usbser_write_done(struct usbser *usbser, struct usbser_wb *wb)
{
	wb->use = 0;
	usbser->transmitting--;
	usb_autopm_put_interface_async(usbser->control);
}

static int usbser_start_wb(struct usbser *usbser, struct usbser_wb *wb)
{
	int rc;

	usbser->transmitting++;

	wb->urb->transfer_buffer = wb->buf;
	wb->urb->transfer_dma = wb->dmah;
	wb->urb->transfer_buffer_length = wb->len;
	wb->urb->dev = usbser->dev;

	rc = usb_submit_urb(wb->urb, GFP_ATOMIC);
	if (rc < 0) {
		dev_err(&usbser->data->dev, "%s - usb_submit_urb(write bulk) failed: %d\n", __func__, rc);
		usbser_write_done(usbser, wb);
	}
	return rc;
}

static void usbser_update_status(struct usbser *usbser, unsigned char *data, size_t len)
{
	unsigned long flags;
	u8 status;
	u8 difference;
	u8 type = data[0];
	u8 handled = 0;

	if (len < 4)
		return;

	if (type & USBSER_CTT_M) {
		status = ~data[len - 1] & USBSER_CTI_ST;
		if (!usbser->clocal && (usbser->ctrlin & status & USBSER_CTI_DC)) {
			tty_port_tty_hangup(&usbser->port, false);
		}

		spin_lock_irqsave(&usbser->read_lock, flags);
		difference = status ^ usbser->ctrlin;
		usbser->ctrlin = status;
		usbser->oldcount = usbser->iocount;

		if (difference) {
			if (difference & USBSER_CTI_C) {
				usbser->iocount.cts++;
			}
			if (difference & USBSER_CTI_DS) {
				usbser->iocount.dsr++;
			}
			if (difference & USBSER_CTI_R) {
				usbser->iocount.rng++;
			}
			if (difference & USBSER_CTI_DC) {
				usbser->iocount.dcd++;
			}
			spin_unlock_irqrestore(&usbser->read_lock, flags);
			wake_up_interruptible(&usbser->wioctl);
		} else
			spin_unlock_irqrestore(&usbser->read_lock, flags);
		handled = 1;
	}
	if (type & USBSER_CTT_O) {
		spin_lock_irqsave(&usbser->read_lock, flags);
		usbser->oldcount = usbser->iocount;
		usbser->iocount.overrun++;
		spin_unlock_irqrestore(&usbser->read_lock, flags);
		handled = 1;
	}
	if ((type & USBSER_CTT_F) == USBSER_CTT_F) {
		spin_lock_irqsave(&usbser->read_lock, flags);
		usbser->oldcount = usbser->iocount;
		usbser->iocount.frame++;
		spin_unlock_irqrestore(&usbser->read_lock, flags);
		handled = 1;
	} else if (type & USBSER_CTT_P) {
		spin_lock_irqsave(&usbser->read_lock, flags);
		usbser->oldcount = usbser->iocount;
		usbser->iocount.parity++;
		spin_unlock_irqrestore(&usbser->read_lock, flags);
		handled = 1;
	}
	if (!handled)
		dev_err(&usbser->control->dev,
			"%s - unknown status received:"
			"len:%d, data0:0x%x, data1:0x%x\n",
			__func__, (int)len, data[0], data[1]);
}

static void usbser_ctrl_irq(struct urb *urb)
{
	struct usbser *usbser = urb->context;
	unsigned char *data = urb->transfer_buffer;
	unsigned int len = urb->actual_length;
	int status = urb->status;
	int retval;

	switch (status) {
	case 0:
		/* success */
		break;
	case -ECONNRESET:
	case -ENOENT:
	case -ESHUTDOWN:
		/* this urb is terminated, clean up */
		dev_dbg(&usbser->control->dev, "%s - urb shutting down with status: %d\n", __func__, status);
		return;
	default:
		dev_dbg(&usbser->control->dev, "%s - nonzero urb status received: %d\n", __func__, status);
		goto exit;
	}

	usb_mark_last_busy(usbser->dev);
	usbser_update_status(usbser, data, len);
exit:
	retval = usb_submit_urb(urb, GFP_ATOMIC);
	if (retval && retval != -EPERM)
		dev_err(&usbser->control->dev, "%s - usb_submit_urb failed: %d\n", __func__, retval);
}

static int usbser_submit_read_urb(struct usbser *usbser, int index, gfp_t mem_flags)
{
	int res;

	if (!test_and_clear_bit(index, &usbser->read_urbs_free))
		return 0;

	res = usb_submit_urb(usbser->read_urbs[index], mem_flags);
	if (res) {
		if (res != -EPERM) {
			dev_err(&usbser->data->dev, "%s - usb_submit_urb failed: %d\n", __func__, res);
		}
		set_bit(index, &usbser->read_urbs_free);
		return res;
	}
	return 0;
}

static int usbser_submit_read_urbs(struct usbser *usbser, gfp_t mem_flags)
{
	int res;
	int i;

	for (i = 0; i < usbser->rx_buflimit; ++i) {
		res = usbser_submit_read_urb(usbser, i, mem_flags);
		if (res)
			return res;
	}
	return 0;
}

static void usbser_process_read_urb(struct usbser *usbser, struct urb *urb)
{
	if (!urb->actual_length)
		return;

	usbser->iocount.rx += urb->actual_length;
	tty_insert_flip_string(&usbser->port, urb->transfer_buffer, urb->actual_length);
	tty_flip_buffer_push(&usbser->port);
}

static void usbser_read_bulk_callback(struct urb *urb)
{
	struct usbser_rb *rb = urb->context;
	struct usbser *usbser = rb->instance;
	int status = urb->status;

	if (!usbser->dev) {
		set_bit(rb->index, &usbser->read_urbs_free);
		dev_dbg(&usbser->data->dev, "%s - disconnected\n", __func__);
		return;
	}

	if (status) {
		set_bit(rb->index, &usbser->read_urbs_free);
		dev_dbg(&usbser->data->dev, "%s - non-zero urb status: %d\n", __func__, status);
		return;
	}

	usb_mark_last_busy(usbser->dev);
	usbser_process_read_urb(usbser, urb);
	set_bit(rb->index, &usbser->read_urbs_free);
	usbser_submit_read_urb(usbser, rb->index, GFP_ATOMIC);
}

static void usbser_write_bulk(struct urb *urb)
{
	struct usbser_wb *wb = urb->context;
	struct usbser *usbser = wb->instance;
	unsigned long flags;
	int status = urb->status;

	if (status || (urb->actual_length != urb->transfer_buffer_length))
		dev_vdbg(&usbser->data->dev, "%s - len %d/%d, status %d\n", __func__, urb->actual_length,
			 urb->transfer_buffer_length, status);

	usbser->iocount.tx += urb->actual_length;
	spin_lock_irqsave(&usbser->write_lock, flags);
	usbser_write_done(usbser, wb);
	spin_unlock_irqrestore(&usbser->write_lock, flags);
	schedule_work(&usbser->work);
}

static void usbser_softint(struct work_struct *work)
{
	struct usbser *usbser = container_of(work, struct usbser, work);
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0))
	struct tty_struct *tty;

	tty = tty_port_tty_get(&usbser->port);
	if (!tty)
		return;
	tty_wakeup(tty);
	tty_kref_put(tty);
#else
	tty_port_tty_wakeup(&usbser->port);
#endif
}

static int usbser_tty_install(struct tty_driver *driver, struct tty_struct *tty)
{
	struct usbser *usbser;
	int retval;

	usbser = usbser_get_by_index(tty->index);
	if (!usbser)
		return -ENODEV;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 5, 0))
	retval = tty_standard_install(driver, tty);
	if (retval)
		goto error_init_termios;

	tty->driver_data = usbser;
#else
	retval = tty_init_termios(tty);
	if (retval)
		goto error_init_termios;

	tty->driver_data = usbser;

	/* Final install (we use the default method) */
	tty_driver_kref_get(driver);
	tty->count++;
	driver->ttys[tty->index] = tty;
#endif

	return 0;

error_init_termios:
	tty_port_put(&usbser->port);
	return retval;
}

static int usbser_tty_open(struct tty_struct *tty, struct file *filp)
{
	struct usbser *usbser = tty->driver_data;

	return tty_port_open(&usbser->port, tty, filp);
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(6, 3, 0))
static void usbser_port_dtr_rts(struct tty_port *port, bool raise)
#else
static void usbser_port_dtr_rts(struct tty_port *port, int raise)
#endif
{
	struct usbser *usbser = container_of(port, struct usbser, port);
	int res;

	if (raise)
		usbser->ctrlout |= USBSER_CTO_D | USBSER_CTO_R;
	else
		usbser->ctrlout &= ~(USBSER_CTO_D | USBSER_CTO_R);

	res = usbser_set_control(usbser, usbser->ctrlout);
	if (res)
		dev_err(&usbser->control->dev, "failed to set dtr/rts\n");
}

static int usbser_port_activate(struct tty_port *port, struct tty_struct *tty)
{
	struct usbser *usbser = container_of(port, struct usbser, port);
	int retval = -ENODEV;
	int i;

	mutex_lock(&usbser->mutex);
	if (usbser->disconnected)
		goto disconnected;

	retval = usb_autopm_get_interface(usbser->control);
	if (retval)
		goto error_get_interface;

	set_bit(TTY_NO_WRITE_SPLIT, &tty->flags);
	usbser->control->needs_remote_wakeup = 1;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 18, 0))
	usbser_tty_set_termios(tty, NULL);
#endif

	retval = usb_submit_urb(usbser->ctrlurb, GFP_KERNEL);
	if (retval) {
		dev_err(&usbser->control->dev, "%s - usb_submit_urb(ctrl cmd) failed\n", __func__);
		goto error_submit_urb;
	}
	retval = usbser_submit_read_urbs(usbser, GFP_KERNEL);
	if (retval)
		goto error_submit_read_urbs;

	usb_autopm_put_interface(usbser->control);
	mutex_unlock(&usbser->mutex);

	return 0;

error_submit_read_urbs:
	for (i = 0; i < usbser->rx_buflimit; i++)
		usb_kill_urb(usbser->read_urbs[i]);
error_submit_urb:
	usb_kill_urb(usbser->ctrlurb);
	usb_autopm_put_interface(usbser->control);
error_get_interface:
disconnected:
	mutex_unlock(&usbser->mutex);
	return usb_translate_errors(retval);
}

static void usbser_port_destruct(struct tty_port *port)
{
	struct usbser *usbser = container_of(port, struct usbser, port);

	usbser_release_minor(usbser);
	usb_put_intf(usbser->control);
	memset(usbser, 0x00, sizeof(struct usbser));
	kfree(usbser);
}

static void usbser_port_shutdown(struct tty_port *port)
{
	struct usbser *usbser = container_of(port, struct usbser, port);
	struct urb *urb;
	struct usbser_wb *wb;
	int i;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 16, 0))
	usb_autopm_get_interface_no_resume(usbser->control);
	usbser->control->needs_remote_wakeup = 0;
	usb_autopm_put_interface(usbser->control);

	for (;;) {
		urb = usb_get_from_anchor(&usbser->delayed);
		if (!urb)
			break;
		wb = urb->context;
		wb->use = 0;
		usb_autopm_put_interface_async(usbser->control);
	}

	usb_kill_urb(usbser->ctrlurb);
	for (i = 0; i < USBSER_NW; i++)
		usb_kill_urb(usbser->wb[i].urb);
	for (i = 0; i < usbser->rx_buflimit; i++)
		usb_kill_urb(usbser->read_urbs[i]);
#else
	mutex_lock(&usbser->mutex);
	if (!usbser->disconnected) {
		usb_autopm_get_interface(usbser->control);
		usbser_set_control(usbser, usbser->ctrlout = 0);

		usb_kill_urb(usbser->ctrlurb);
		for (i = 0; i < USBSER_NW; i++)
			usb_kill_urb(usbser->wb[i].urb);
		for (i = 0; i < usbser->rx_buflimit; i++)
			usb_kill_urb(usbser->read_urbs[i]);
		usbser->control->needs_remote_wakeup = 0;

		usb_autopm_put_interface(usbser->control);
	}
	mutex_unlock(&usbser->mutex);
#endif
}

static void usbser_tty_cleanup(struct tty_struct *tty)
{
	struct usbser *usbser = tty->driver_data;
	tty_port_put(&usbser->port);
}

static void usbser_tty_hangup(struct tty_struct *tty)
{
	struct usbser *usbser = tty->driver_data;
	tty_port_hangup(&usbser->port);
}

static void usbser_tty_close(struct tty_struct *tty, struct file *filp)
{
	struct usbser *usbser = tty->driver_data;
	tty_port_close(&usbser->port, tty, filp);
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(6, 6, 0))
static ssize_t usbser_tty_write(struct tty_struct *tty, const u8 *buf, size_t count)
#else
static int usbser_tty_write(struct tty_struct *tty, const unsigned char *buf, int count)
#endif
{
	struct usbser *usbser = tty->driver_data;
	int stat;
	unsigned long flags;
	int wbn;
	struct usbser_wb *wb;

	if (!count)
		return 0;

	spin_lock_irqsave(&usbser->write_lock, flags);
	wbn = usbser_wb_alloc(usbser);
	if (wbn < 0) {
		spin_unlock_irqrestore(&usbser->write_lock, flags);
		return 0;
	}
	wb = &usbser->wb[wbn];

	if (!usbser->dev) {
		wb->use = 0;
		spin_unlock_irqrestore(&usbser->write_lock, flags);
		return -ENODEV;
	}

	count = (count > usbser->writesize) ? usbser->writesize : count;

	memcpy(wb->buf, buf, count);
	wb->len = count;

	stat = usb_autopm_get_interface_async(usbser->control);
	if (stat) {
		wb->use = 0;
		spin_unlock_irqrestore(&usbser->write_lock, flags);
		return stat;
	}

	if (usbser->susp_count) {
		usb_anchor_urb(wb->urb, &usbser->delayed);
		spin_unlock_irqrestore(&usbser->write_lock, flags);
		return count;
	}

	stat = usbser_start_wb(usbser, wb);
	spin_unlock_irqrestore(&usbser->write_lock, flags);

	if (stat < 0)
		return stat;
	return count;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0))
static unsigned int usbser_tty_write_room(struct tty_struct *tty)
#else
static int usbser_tty_write_room(struct tty_struct *tty)
#endif
{
	struct usbser *usbser = tty->driver_data;

	return usbser_wb_is_avail(usbser) ? usbser->writesize : 0;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0))
static unsigned int usbser_tty_chars_in_buffer(struct tty_struct *tty)
#else
static int usbser_tty_chars_in_buffer(struct tty_struct *tty)
#endif
{
	struct usbser *usbser = tty->driver_data;

	if (usbser->disconnected)
		return 0;

	return (USBSER_NW - usbser_wb_is_avail(usbser)) * usbser->writesize;
}

static int usbser_tty_break_ctl(struct tty_struct *tty, int state)
{
	struct usbser *usbser = tty->driver_data;
	int retval;
	uint16_t reg_contents;
	uint8_t *regbuf;

	regbuf = kmalloc(2, GFP_KERNEL);
	if (!regbuf)
		return -1;

	if (state != 0) {
		regbuf[0] = USBSER_N_B;
		regbuf[1] = 0x00;
	} else {
		regbuf[0] = USBSER_N_B | USBSER_N_AB;
		regbuf[1] = 0x00;
	}
	reg_contents = get_unaligned_le16(regbuf);
	if (usbser->iface)
		retval = usbser_control_out(usbser, CMD_C4, 0x00, reg_contents);
	else
		retval = usbser_control_out(usbser, CMD_C4, reg_contents, 0x00);

	if (retval < 0)
		dev_err(&usbser->control->dev, "%s - USB control write error (%d)\n", __func__, retval);

	kfree(regbuf);
	return retval;
}

static int usbser_tty_tiocmget(struct tty_struct *tty)
{
	struct usbser *usbser = tty->driver_data;
	unsigned long flags;
	unsigned int result;

	spin_lock_irqsave(&usbser->read_lock, flags);
	result = (usbser->ctrlout & USBSER_CTO_D ? TIOCM_DTR : 0) | (usbser->ctrlout & USBSER_CTO_R ? TIOCM_RTS : 0) |
		 (usbser->ctrlin & USBSER_CTI_C ? TIOCM_CTS : 0) | (usbser->ctrlin & USBSER_CTI_DS ? TIOCM_DSR : 0) |
		 (usbser->ctrlin & USBSER_CTI_R ? TIOCM_RI : 0) | (usbser->ctrlin & USBSER_CTI_DC ? TIOCM_CD : 0);
	spin_unlock_irqrestore(&usbser->read_lock, flags);

	return result;
}

static int usbser_tty_tiocmset(struct tty_struct *tty, unsigned int set, unsigned int clear)
{
	struct usbser *usbser = tty->driver_data;
	unsigned int newctrl;

	newctrl = usbser->ctrlout;
	set = (set & TIOCM_DTR ? USBSER_CTO_D : 0) | (set & TIOCM_RTS ? USBSER_CTO_R : 0);
	clear = (clear & TIOCM_DTR ? USBSER_CTO_D : 0) | (clear & TIOCM_RTS ? USBSER_CTO_R : 0);

	newctrl = (newctrl & ~clear) | set;

	if (usbser->ctrlout == newctrl) {
		return 0;
	}

	return usbser_set_control(usbser, usbser->ctrlout = newctrl);
}

static int usbser_get_icount(struct tty_struct *tty, struct serial_icounter_struct *icount)
{
	struct usbser *usbser = tty->driver_data;
	struct async_icount cnow;
	unsigned long flags;

	spin_lock_irqsave(&usbser->read_lock, flags);
	cnow = usbser->iocount;
	spin_unlock_irqrestore(&usbser->read_lock, flags);

	icount->cts = cnow.cts;
	icount->dsr = cnow.dsr;
	icount->rng = cnow.rng;
	icount->dcd = cnow.dcd;
	icount->tx = cnow.tx;
	icount->rx = cnow.rx;
	icount->frame = cnow.frame;
	icount->parity = cnow.parity;
	icount->overrun = cnow.overrun;
	icount->brk = cnow.brk;
	icount->buf_overrun = cnow.buf_overrun;

	return 0;
}

static int usbser_get_serial_info(struct usbser *usbser, struct serial_struct __user *info)
{
	struct serial_struct tmp;

	if (!info)
		return -EINVAL;

	memset(&tmp, 0, sizeof(tmp));
	tmp.flags = ASYNC_LOW_LATENCY;
	tmp.xmit_fifo_size = usbser->writesize;
	tmp.baud_base = le32_to_cpu(usbser->line.dwDTERate);
	tmp.close_delay = usbser->port.close_delay / 10;
	tmp.closing_wait = usbser->port.closing_wait == ASYNC_CLOSING_WAIT_NONE ? ASYNC_CLOSING_WAIT_NONE :
										 usbser->port.closing_wait / 10;

	if (copy_to_user(info, &tmp, sizeof(tmp)))
		return -EFAULT;
	else
		return 0;
}

static int usbser_set_serial_info(struct usbser *usbser, struct serial_struct __user *newinfo)
{
	struct serial_struct new_serial;
	unsigned int closing_wait, close_delay;
	int retval = 0;

	if (copy_from_user(&new_serial, newinfo, sizeof(new_serial)))
		return -EFAULT;

	close_delay = new_serial.close_delay * 10;
	closing_wait = new_serial.closing_wait == ASYNC_CLOSING_WAIT_NONE ? ASYNC_CLOSING_WAIT_NONE :
									    new_serial.closing_wait * 10;

	mutex_lock(&usbser->port.mutex);

	if (!capable(CAP_SYS_ADMIN)) {
		if ((close_delay != usbser->port.close_delay) || (closing_wait != usbser->port.closing_wait))
			retval = -EPERM;
		else
			retval = -EOPNOTSUPP;
	} else {
		usbser->port.close_delay = close_delay;
		usbser->port.closing_wait = closing_wait;
	}

	mutex_unlock(&usbser->port.mutex);
	return retval;
}

static int usbser_wait_serial_change(struct usbser *usbser, unsigned long arg)
{
	int rv = 0;
	DECLARE_WAITQUEUE(wait, current);
	struct async_icount old, new;

	do {
		spin_lock_irq(&usbser->read_lock);
		old = usbser->oldcount;
		new = usbser->iocount;
		usbser->oldcount = new;
		spin_unlock_irq(&usbser->read_lock);

		if ((arg & TIOCM_CTS) && old.cts != new.cts)
			break;
		if ((arg & TIOCM_DSR) && old.dsr != new.dsr)
			break;
		if ((arg & TIOCM_RI) && old.rng != new.rng)
			break;
		if ((arg & TIOCM_CD) && old.dcd != new.dcd)
			break;

		add_wait_queue(&usbser->wioctl, &wait);
		set_current_state(TASK_INTERRUPTIBLE);
		schedule();
		remove_wait_queue(&usbser->wioctl, &wait);
		if (usbser->disconnected) {
			if (arg & TIOCM_CD)
				break;
			else
				rv = -ENODEV;
		} else {
			if (signal_pending(current))
				rv = -ERESTARTSYS;
		}
	} while (!rv);

	return rv;
}

static int usbser_get_serial_usage(struct usbser *usbser, struct serial_icounter_struct __user *count)
{
	struct serial_icounter_struct icount;
	int rv = 0;

	memset(&icount, 0, sizeof(icount));
	icount.cts = usbser->iocount.cts;
	icount.dsr = usbser->iocount.dsr;
	icount.rng = usbser->iocount.rng;
	icount.dcd = usbser->iocount.dcd;
	icount.tx = usbser->iocount.tx;
	icount.rx = usbser->iocount.rx;
	icount.frame = usbser->iocount.frame;
	icount.overrun = usbser->iocount.overrun;
	icount.parity = usbser->iocount.parity;
	icount.brk = usbser->iocount.brk;
	icount.buf_overrun = usbser->iocount.buf_overrun;

	if (copy_to_user(count, &icount, sizeof(icount)) > 0)
		rv = -EFAULT;

	return rv;
}

static int usbser_tty_ioctl(struct tty_struct *tty, unsigned int cmd, unsigned long arg)
{
	struct usbser *usbser = tty->driver_data;
	int rv = 0;
	unsigned long arg1, arg2, arg3, arg4, arg5, arg6;
	u32 __user *argval = (u32 __user *)arg;
	u8 *buffer;

	buffer = kmalloc(512, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	switch (cmd) {
	case TIOCGSERIAL: /* gets serial port data */
		rv = usbser_get_serial_info(usbser, (struct serial_struct __user *)arg);
		break;
	case TIOCSSERIAL:
		rv = usbser_set_serial_info(usbser, (struct serial_struct __user *)arg);
		break;
	case TIOCMIWAIT:
		rv = usb_autopm_get_interface(usbser->control);
		if (rv < 0) {
			rv = -EIO;
			break;
		}
		rv = usbser_wait_serial_change(usbser, arg);
		usb_autopm_put_interface(usbser->control);
		break;
	case TIOCGICOUNT:
		rv = usbser_get_serial_usage(usbser, (struct serial_icounter_struct __user *)arg);
		break;
	case IOCTL_CMD_GETCHIPTYPE:
		if (put_user(usbser->chiptype, argval)) {
			rv = -EFAULT;
			goto out;
		}
		break;
	case IOCTL_CMD_GETUARTINDEX:
		if (put_user(usbser->iface, argval)) {
			rv = -EFAULT;
			goto out;
		}
		break;
	case IOCTL_CMD_CTRLIN:
		get_user(arg1, (u8 __user *)arg);
		get_user(arg2, ((u8 __user *)arg + 1));
		get_user(arg3, (u16 __user *)((u8 *)arg + 2));
		get_user(arg4, (u16 __user *)((u8 *)arg + 4));
		get_user(arg5, (u16 __user *)((u8 *)arg + 6));
		arg6 = (unsigned long)((u8 __user *)arg + 8);
		rv = usbser_control_msg_in(usbser, (u8)arg1, (u8)arg2, (u16)arg3, (u16)arg4, (u8 __user *)arg6,
					  (u16)arg5);
		break;
	case IOCTL_CMD_CTRLOUT:
		get_user(arg1, (u8 __user *)arg);
		get_user(arg2, ((u8 __user *)arg + 1));
		get_user(arg3, (u16 __user *)((u8 *)arg + 2));
		get_user(arg4, (u16 __user *)((u8 *)arg + 4));
		get_user(arg5, (u16 __user *)((u8 *)arg + 6));
		arg6 = (unsigned long)((u8 __user *)arg + 8);
		rv = usbser_control_msg_out(usbser, (u8)arg1, (u8)arg2, (u16)arg3, (u16)arg4, (u8 __user *)arg6,
					   (u16)arg5);
		if (rv != (u16)arg5) {
			rv = -EINVAL;
			goto out;
		}
		break;
	default:
		rv = -ENOIOCTLCMD;
		break;
	}
out:
	kfree(buffer);
	return rv;
}

static int usbser_get(CHIPTYPE chiptype, unsigned int bval, unsigned char *fct, unsigned char *dvs)
{
	unsigned char a;
	unsigned char b;
	unsigned long c;

	switch (bval) {
	case 6000000:
	case 4000000:
	case 2400000:
	case 921600:
	case 307200:
	case 256000:
		b = 7;
		c = 12000000;
		break;
	default:
		if (bval > 6000000 / 255) {
			b = 3;
			c = 6000000;
		} else if (bval > 750000 / 255) {
			b = 2;
			c = 750000;
		} else if (bval > 93750 / 255) {
			b = 1;
			c = 93750;
		} else {
			b = 0;
			c = 11719;
		}
		break;
	}
	a = (unsigned char)(c / bval);
	if (a == 0 || a == 0xFF)
		return -EINVAL;
	if ((c / a - bval) > (bval - c / (a + 1)))
		a++;
	a = 256 - a;

	*fct = a;
	*dvs = b;

	return 0;
}

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(6, 1, 0))
static void usbser_tty_set_termios(struct tty_struct *tty, const struct ktermios *termios_old)
#else
static void usbser_tty_set_termios(struct tty_struct *tty, struct ktermios *termios_old)
#endif
{
	struct usbser *usbser = tty->driver_data;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 7, 0))
	struct ktermios *termios = &tty->termios;
#else
	struct ktermios *termios = tty->termios;
#endif
	struct usb_usbser_line_coding newline;
	int newctrl = usbser->ctrlout;

	unsigned char dvs = 0;
	unsigned char reg_count = 0;
	unsigned char fct = 0;
	unsigned char reg_value = 0;
	unsigned short value = 0;
	unsigned short index = 0;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 7, 0))
	if (termios_old && !tty_termios_hw_change(&tty->termios, termios_old)) {
#else
	if (termios_old && !tty_termios_hw_change(tty->termios, termios_old)) {
#endif
		return;
	}

	newline.dwDTERate = tty_get_baud_rate(tty);

	if (newline.dwDTERate == 0)
		newline.dwDTERate = 9600;
	usbser_get(usbser->chiptype, newline.dwDTERate, &fct, &dvs);

	newline.bCharFormat = termios->c_cflag & CSTOPB ? 2 : 1;
	if (newline.bCharFormat == 2)
		reg_value |= USBSER_L_SB;

	newline.bParityType = termios->c_cflag & PARENB ?
				      (termios->c_cflag & PARODD ? 1 : 2) + (termios->c_cflag & CMSPAR ? 2 : 0) :
				      0;

	switch (newline.bParityType) {
	case 0x01:
		reg_value |= USBSER_L_P_O;
		break;
	case 0x02:
		reg_value |= USBSER_L_P_E;
		break;
	case 0x03:
		reg_value |= USBSER_L_P_M;
		break;
	case 0x04:
		reg_value |= USBSER_L_P_S;
		break;
	default:
		break;
	}

	switch (termios->c_cflag & CSIZE) {
	case CS5:
		newline.bDataBits = 5;
		reg_value |= USBSER_L_C5;
		break;
	case CS6:
		newline.bDataBits = 6;
		reg_value |= USBSER_L_C6;
		break;
	case CS7:
		newline.bDataBits = 7;
		reg_value |= USBSER_L_C7;
		break;
	case CS8:
	default:
		newline.bDataBits = 8;
		reg_value |= USBSER_L_C8;
		break;
	}

	usbser->clocal = ((termios->c_cflag & CLOCAL) != 0);

	if (C_BAUD(tty) == B0) {
		newline.dwDTERate = usbser->line.dwDTERate;
		newctrl &= ~USBSER_CTO_D;
	} else if (termios_old && (termios_old->c_cflag & CBAUD) == B0) {
		newctrl |= USBSER_CTO_D;
	}

	reg_value |= USBSER_L_E_R | USBSER_L_E_T;
	reg_count |= USBSER_L_R_CT | USBSER_L_R_CL | USBSER_L_R_T;

	value |= reg_count;
	value |= (unsigned short)reg_value << 8;

	index |= 0x00 | dvs;
	index |= (unsigned short)fct << 8;
	if (usbser->iface <= 1)
		usbser_control_out(usbser, CMD_C1 + usbser->iface, value, index);
	else if (usbser->iface <= 3)
		usbser_control_out(usbser, CMD_C1 + 0x10 + (usbser->iface - 2), value, index);

	if (memcmp(&usbser->line, &newline, sizeof newline))
		memcpy(&usbser->line, &newline, sizeof newline);

	if (C_CRTSCTS(tty)) {
		newctrl |= USBSER_CTO_A | USBSER_CTO_R;
	} else
		newctrl &= ~USBSER_CTO_A;

	if (newctrl != usbser->ctrlout)
		usbser_set_control(usbser, usbser->ctrlout = newctrl);
}

static const struct tty_port_operations usbser_port_ops = {
	.dtr_rts = usbser_port_dtr_rts,
	.shutdown = usbser_port_shutdown,
	.activate = usbser_port_activate,
	.destruct = usbser_port_destruct,
};

static void usbser_write_buffers_free(struct usbser *usbser)
{
	int i;
	struct usbser_wb *wb;
	struct usb_device *usb_dev = interface_to_usbdev(usbser->control);

	for (wb = &usbser->wb[0], i = 0; i < USBSER_NW; i++, wb++)
		usb_free_coherent(usb_dev, usbser->writesize, wb->buf, wb->dmah);
}

static void usbser_read_buffers_free(struct usbser *usbser)
{
	struct usb_device *usb_dev = interface_to_usbdev(usbser->control);
	int i;

	for (i = 0; i < usbser->rx_buflimit; i++)
		usb_free_coherent(usb_dev, usbser->readsize, usbser->read_buffers[i].base, usbser->read_buffers[i].dma);
}

static int usbser_write_buffers_alloc(struct usbser *usbser)
{
	int i;
	struct usbser_wb *wb;

	for (wb = &usbser->wb[0], i = 0; i < USBSER_NW; i++, wb++) {
		wb->buf = usb_alloc_coherent(usbser->dev, usbser->writesize, GFP_KERNEL, &wb->dmah);
		if (!wb->buf) {
			while (i != 0) {
				--i;
				--wb;
				usb_free_coherent(usbser->dev, usbser->writesize, wb->buf, wb->dmah);
			}
			return -ENOMEM;
		}
	}
	return 0;
}

static int usbser_open(struct inode *inode, struct file *file)
{
	struct usbser *usbser;
	struct usb_interface *interface;
	int subminor;
	int retval = 0;

	subminor = iminor(inode);

	interface = usb_find_interface(&usbser_driver, subminor);
	if (!interface) {
		retval = -ENODEV;
		goto exit;
	}

	usbser = usb_get_intfdata(interface);
	if (!usbser) {
		retval = -ENODEV;
		goto exit;
	}

	file->private_data = usbser;

exit:
	return retval;
}

static int usbser_release(struct inode *inode, struct file *file)
{
	struct usbser *usbser;

	usbser = file->private_data;
	if (usbser == NULL || usbser->io_id != IOID)
		return -ENODEV;

	return 0;
}

static long usbser_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct usbser *usbser;
	int rv = 0;
	u8 *buffer;
	unsigned long arg1, arg2, arg3, arg4, arg5, arg6;
	u32 __user *argval = (u32 __user *)arg;

	usbser = file->private_data;
	if (usbser == NULL || usbser->io_id != IOID)
		return -ENODEV;

	buffer = kmalloc(512, GFP_KERNEL);
	if (!buffer)
		return -ENOMEM;

	switch (cmd) {
	case IOCTL_CMD_GETCHIPTYPE:
		if (put_user(usbser->chiptype, argval)) {
			rv = -EFAULT;
			goto out;
		}
		break;
	case IOCTL_CMD_CTRLIN:
		get_user(arg1, (u8 __user *)arg);
		get_user(arg2, ((u8 __user *)arg + 1));
		get_user(arg3, (u16 __user *)((u8 *)arg + 2));
		get_user(arg4, (u16 __user *)((u8 *)arg + 4));
		get_user(arg5, (u16 __user *)((u8 *)arg + 6));
		arg6 = (unsigned long)((u8 __user *)arg + 8);
		rv = usbser_control_msg_in(usbser, (u8)arg1, (u8)arg2, (u16)arg3, (u16)arg4, (u8 __user *)arg6,
					  (u16)arg5);
		break;
	case IOCTL_CMD_CTRLOUT:
		get_user(arg1, (u8 __user *)arg);
		get_user(arg2, ((u8 __user *)arg + 1));
		get_user(arg3, (u16 __user *)((u8 *)arg + 2));
		get_user(arg4, (u16 __user *)((u8 *)arg + 4));
		get_user(arg5, (u16 __user *)((u8 *)arg + 6));
		arg6 = (unsigned long)((u8 __user *)arg + 8);
		rv = usbser_control_msg_out(usbser, (u8)arg1, (u8)arg2, (u16)arg3, (u16)arg4, (u8 __user *)arg6,
					   (u16)arg5);
		if (rv != (u16)arg5) {
			rv = -EINVAL;
			goto out;
		}
		break;
	default:
		rv = -ENOIOCTLCMD;
		break;
	}

out:
	kfree(buffer);
	return rv;
}

#ifdef CONFIG_COMPAT
static long usbser_compat_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	return usbser_ioctl(file, cmd, arg);
}
#endif

static const struct file_operations usbser_fops = {
	.owner = THIS_MODULE,
	.open = usbser_open,
	.unlocked_ioctl = usbser_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = usbser_compat_ioctl,
#endif
	.release = usbser_release,
};

/*
 * usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with the driver core
 */
static struct usb_class_driver usbser_class = {
	.name = "usbser_iodev%d",
	.fops = &usbser_fops,
	.minor_base = USB_MINOR_BASE,
};

/*
 * USB probe and disconnect routines.
 */
static int usbser_probe(struct usb_interface *intf, const struct usb_device_id *id)
{
	struct usb_cdc_union_desc *union_header = NULL;
	unsigned char *buffer = intf->altsetting->extra;
	int buflen = intf->altsetting->extralen;
	struct usb_interface *control_interface;
	struct usb_interface *data_interface;
	struct usb_endpoint_descriptor *epctrl = NULL;
	struct usb_endpoint_descriptor *epread = NULL;
	struct usb_endpoint_descriptor *epwrite = NULL;
	struct usb_device *usb_dev = interface_to_usbdev(intf);
	struct usbser *usbser;
	int minor;
	int ctrlsize, readsize;
	u8 *buf;
	unsigned long quirks;
	int num_rx_buf = USBSER_NR;
	int i;
	unsigned int elength = 0;
	struct device *tty_dev;
	int rv = -ENOMEM;

	quirks = (unsigned long)id->driver_info;
	if ((!buffer) || (buflen == 0)) {
		dev_err(&intf->dev, "Weird descriptor references\n");
		return -EINVAL;
	}

	while (buflen > 0) {
		elength = buffer[0];
		if (!elength) {
			dev_err(&intf->dev, "skipping garbage byte\n");
			elength = 1;
			goto next_desc;
		}
		if (buffer[1] != USB_DT_CS_INTERFACE) {
			dev_err(&intf->dev, "skipping garbage\n");
			goto next_desc;
		}

		switch (buffer[2]) {
		case USB_CDC_UNION_TYPE:
			if (elength < sizeof(struct usb_cdc_union_desc))
				goto next_desc;
			if (union_header) {
				dev_err(&intf->dev, "More than one "
						    "union descriptor, skipping ...\n");
				goto next_desc;
			}
			union_header = (struct usb_cdc_union_desc *)buffer;
			break;
		default:
			/*
             * there are LOTS more CDC descriptors that
             * could legitimately be found here.
             */
			break;
		}
next_desc:
		buflen -= elength;
		buffer += elength;
	}

	if (!union_header) {
		dev_err(&intf->dev, "Weird descriptor references\n");
		return -EINVAL;
	}

	control_interface = usb_ifnum_to_if(usb_dev, union_header->bMasterInterface0);
	data_interface = usb_ifnum_to_if(usb_dev, union_header->bSlaveInterface0);

	if (intf != control_interface)
		return -ENODEV;

	if (usb_interface_claimed(data_interface)) {
		dev_err(&intf->dev, "The data interface isn't available\n");
		return -EBUSY;
	}

	if (data_interface->cur_altsetting->desc.bNumEndpoints < 2 ||
	    control_interface->cur_altsetting->desc.bNumEndpoints == 0)
		return -EINVAL;

	epctrl = &control_interface->cur_altsetting->endpoint[0].desc;
	epwrite = &data_interface->cur_altsetting->endpoint[0].desc;
	epread = &data_interface->cur_altsetting->endpoint[1].desc;

	if (!usb_endpoint_dir_in(epread))
		swap(epread, epwrite);

	usbser = kzalloc(sizeof(struct usbser), GFP_KERNEL);
	if (!usbser)
		return -ENOMEM;

	usbser->idVendor = id->idVendor;
	usbser->idProduct = id->idProduct;
	usbser->iface = control_interface->cur_altsetting->desc.bInterfaceNumber / 2;
	
	usb_get_intf(control_interface);
	minor = usbser_alloc_minor(usbser);
	if (minor < 0) {
		dev_err(&intf->dev, "no more free usbser devices\n");
		usbser->minor = USBSER_MINOR_INVALID;
		goto alloc_fail;
	}

	ctrlsize = usb_endpoint_maxp(epctrl);
	readsize = usb_endpoint_maxp(epread);
	usbser->writesize = usb_endpoint_maxp(epwrite) * 20;
	usbser->control = control_interface;
	usbser->data = data_interface;
	usbser->minor = minor;
	usbser->dev = usb_dev;
	usbser->ctrlsize = ctrlsize;
	usbser->readsize = readsize;
	usbser->rx_buflimit = num_rx_buf;

	INIT_WORK(&usbser->work, usbser_softint);
	init_waitqueue_head(&usbser->wioctl);
	spin_lock_init(&usbser->write_lock);
	spin_lock_init(&usbser->read_lock);
	mutex_init(&usbser->mutex);
	usbser->rx_endpoint = usb_rcvbulkpipe(usb_dev, epread->bEndpointAddress);
	tty_port_init(&usbser->port);
	usbser->port.ops = &usbser_port_ops;
	init_usb_anchor(&usbser->delayed);
	usbser->quirks = quirks;

	buf = usb_alloc_coherent(usb_dev, ctrlsize, GFP_KERNEL, &usbser->ctrl_dma);
	if (!buf)
		goto err_put_port;
	usbser->ctrl_buffer = buf;

	if (usbser_write_buffers_alloc(usbser) < 0)
		goto err_free_ctrl_buffer;

	usbser->ctrlurb = usb_alloc_urb(0, GFP_KERNEL);
	if (!usbser->ctrlurb)
		goto err_free_write_buffers;

	for (i = 0; i < num_rx_buf; i++) {
		struct usbser_rb *rb = &(usbser->read_buffers[i]);
		struct urb *urb;

		rb->base = usb_alloc_coherent(usbser->dev, readsize, GFP_KERNEL, &rb->dma);
		if (!rb->base)
			goto err_free_read_urbs;
		rb->index = i;
		rb->instance = usbser;

		urb = usb_alloc_urb(0, GFP_KERNEL);
		if (!urb)
			goto err_free_read_urbs;

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = rb->dma;
		usb_fill_bulk_urb(urb, usbser->dev, usbser->rx_endpoint, rb->base, usbser->readsize,
				  usbser_read_bulk_callback, rb);

		usbser->read_urbs[i] = urb;
		__set_bit(i, &usbser->read_urbs_free);
	}
	for (i = 0; i < USBSER_NW; i++) {
		struct usbser_wb *snd = &(usbser->wb[i]);

		snd->urb = usb_alloc_urb(0, GFP_KERNEL);
		if (snd->urb == NULL)
			goto err_free_write_urbs;

		usb_fill_bulk_urb(snd->urb, usb_dev, usb_sndbulkpipe(usb_dev, epwrite->bEndpointAddress), NULL,
				  usbser->writesize, usbser_write_bulk, snd);
		snd->urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		snd->instance = usbser;
	}

	usb_set_intfdata(intf, usbser);

	usb_fill_int_urb(usbser->ctrlurb, usb_dev, usb_rcvintpipe(usb_dev, epctrl->bEndpointAddress), usbser->ctrl_buffer,
			 ctrlsize, usbser_ctrl_irq, usbser, epctrl->bInterval ? epctrl->bInterval : 16);
	usbser->ctrlurb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
	usbser->ctrlurb->transfer_dma = usbser->ctrl_dma;

	dev_info(&intf->dev, "ttyUSBSER%d: usb to uart device\n", minor);

	rv = usbser_configure(usbser);
	if (rv)
		goto err_free_write_urbs;

	if (usbser->iosupport && (usbser->iface == 0) && (usbser->io_intf == NULL)) {
		/* register the device now, as it is ready */
		rv = usb_register_dev(intf, &usbser_class);
		if (rv) {
			/* error when registering this driver */
			dev_err(&intf->dev, "Not able to get a minor for this device.\n");
		} else {
			usbser->io_id = IOID;
			usbser->io_intf = intf;
			dev_info(&intf->dev, "USB to GPIO device now attached to usbser_iodev%d\n", intf->minor);
		}
	}

	usb_driver_claim_interface(&usbser_driver, data_interface, usbser);
	usb_set_intfdata(data_interface, usbser);

	usbser->line.dwDTERate = 9600;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 7, 0))
	tty_dev = tty_port_register_device(&usbser->port, usbser_tty_driver, minor, &control_interface->dev);
	if (IS_ERR(tty_dev)) {
		rv = PTR_ERR(tty_dev);
		goto err_release_data_interface;
	}
#else
	tty_register_device(usbser_tty_driver, minor, &control_interface->dev);
#endif

	return 0;

err_release_data_interface:
	usb_set_intfdata(data_interface, NULL);
	usb_driver_release_interface(&usbser_driver, data_interface);
err_free_write_urbs:
	for (i = 0; i < USBSER_NW; i++)
		usb_free_urb(usbser->wb[i].urb);
err_free_read_urbs:
	for (i = 0; i < num_rx_buf; i++)
		usb_free_urb(usbser->read_urbs[i]);
	usbser_read_buffers_free(usbser);
	usb_free_urb(usbser->ctrlurb);
err_free_write_buffers:
	usbser_write_buffers_free(usbser);
err_free_ctrl_buffer:
	usb_free_coherent(usb_dev, ctrlsize, usbser->ctrl_buffer, usbser->ctrl_dma);
err_put_port:
	tty_port_put(&usbser->port);
alloc_fail:
	kfree(usbser);

	return rv;
}

static void stop_data_traffic(struct usbser *usbser)
{
	struct urb *urb;
	struct usbser_wb *wb;
	int i;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 16, 0))

	usb_autopm_get_interface_no_resume(usbser->control);
	usbser->control->needs_remote_wakeup = 0;
	usb_autopm_put_interface(usbser->control);

	for (;;) {
		urb = usb_get_from_anchor(&usbser->delayed);
		if (!urb)
			break;
		wb = urb->context;
		wb->use = 0;
		usb_autopm_put_interface_async(usbser->control);
	}

	usb_kill_urb(usbser->ctrlurb);
	for (i = 0; i < USBSER_NW; i++)
		usb_kill_urb(usbser->wb[i].urb);
	for (i = 0; i < usbser->rx_buflimit; i++)
		usb_kill_urb(usbser->read_urbs[i]);

#else
	mutex_lock(&usbser->mutex);
	if (!usbser->disconnected) {
		usb_autopm_get_interface(usbser->control);
		usbser_set_control(usbser, usbser->ctrlout = 0);

		usb_kill_urb(usbser->ctrlurb);
		for (i = 0; i < USBSER_NW; i++)
			usb_kill_urb(usbser->wb[i].urb);
		for (i = 0; i < usbser->rx_buflimit; i++)
			usb_kill_urb(usbser->read_urbs[i]);
		usbser->control->needs_remote_wakeup = 0;

		usb_autopm_put_interface(usbser->control);
	}
	mutex_unlock(&usbser->mutex);
#endif
}

static void usbser_disconnect(struct usb_interface *intf)
{
	struct usbser *usbser = usb_get_intfdata(intf);
	struct usb_device *usb_dev = interface_to_usbdev(intf);
	struct tty_struct *tty;
	int i;

	/* sibling interface is already cleaning up */
	if (!usbser)
		return;

	/* give back minor */
	if (usbser->iosupport && (usbser->iface == 0) && (usbser->io_intf != NULL)) {
		usb_deregister_dev(usbser->io_intf, &usbser_class);
		usbser->io_intf = NULL;
	}

	mutex_lock(&usbser->mutex);
	usbser->disconnected = true;
	wake_up_all(&usbser->wioctl);
	usb_set_intfdata(usbser->control, NULL);
	usb_set_intfdata(usbser->data, NULL);
	mutex_unlock(&usbser->mutex);

	tty = tty_port_tty_get(&usbser->port);
	if (tty) {
		tty_vhangup(tty);
		tty_kref_put(tty);
	}

	stop_data_traffic(usbser);
	tty_unregister_device(usbser_tty_driver, usbser->minor);

	usb_free_urb(usbser->ctrlurb);
	for (i = 0; i < USBSER_NW; i++)
		usb_free_urb(usbser->wb[i].urb);
	for (i = 0; i < usbser->rx_buflimit; i++)
		usb_free_urb(usbser->read_urbs[i]);
	usbser_write_buffers_free(usbser);
	usb_free_coherent(usb_dev, usbser->ctrlsize, usbser->ctrl_buffer, usbser->ctrl_dma);
	usbser_read_buffers_free(usbser);

	usb_driver_release_interface(&usbser_driver, intf == usbser->control ? usbser->data : usbser->control);
	tty_port_put(&usbser->port);
	dev_info(&intf->dev, "%s\n", "usbser usb device disconnect.");
}

#ifdef CONFIG_PM
static int usbser_suspend(struct usb_interface *intf, pm_message_t message)
{
	struct usbser *usbser = usb_get_intfdata(intf);
	int cnt;

	spin_lock_irq(&usbser->write_lock);
	if (PMSG_IS_AUTO(message)) {
		if (usbser->transmitting) {
			spin_unlock_irq(&usbser->write_lock);
			return -EBUSY;
		}
	}
	cnt = usbser->susp_count++;
	spin_unlock_irq(&usbser->write_lock);
	if (cnt)
		return 0;
	stop_data_traffic(usbser);

	return 0;
}

static int usbser_resume(struct usb_interface *intf)
{
	struct usbser *usbser = usb_get_intfdata(intf);
	struct urb *urb;
	int rv = 0;

	spin_lock_irq(&usbser->write_lock);
	if (--usbser->susp_count)
		goto out;

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 7, 0))
	if (tty_port_initialized(&usbser->port)) {
#else
	if (test_bit(ASYNCB_INITIALIZED, &usbser->port.flags)) {
#endif
		rv = usb_submit_urb(usbser->ctrlurb, GFP_ATOMIC);
		for (;;) {
			urb = usb_get_from_anchor(&usbser->delayed);
			if (!urb)
				break;

			usbser_start_wb(usbser, urb->context);
		}
		if (rv < 0)
			goto out;
		rv = usbser_submit_read_urbs(usbser, GFP_ATOMIC);
	}
out:
	spin_unlock_irq(&usbser->write_lock);
	return rv;
}

static int usbser_reset_resume(struct usb_interface *intf)
{
	struct usbser *usbser = usb_get_intfdata(intf);

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(4, 7, 0))
	if (tty_port_initialized(&usbser->port))
#else
	if (test_bit(ASYNCB_INITIALIZED, &usbser->port.flags))
#endif
		tty_port_tty_hangup(&usbser->port, false);

	return usbser_resume(intf);
}
#endif /* CONFIG_PM */

static const struct usb_device_id usbser_ids[] = {
	{ USB_DEVICE(0x1a86, 0xe203) },			       /* usbserial chip */
	{}
};

MODULE_DEVICE_TABLE(usb, usbser_ids);

static struct usb_driver usbser_driver = {
	.name = "usb_ztekser",
	.probe = usbser_probe,
	.disconnect = usbser_disconnect,
#ifdef CONFIG_PM
	.suspend = usbser_suspend,
	.resume = usbser_resume,
	.reset_resume = usbser_reset_resume,
#endif
	.id_table = usbser_ids,
#ifdef CONFIG_PM
	.supports_autosuspend = 1,
#endif
	.disable_hub_initiated_lpm = 1,
};

/*
 * TTY driver structures.
 */
static const struct tty_operations usbser_ops = {
	.install = usbser_tty_install,
	.open = usbser_tty_open,
	.close = usbser_tty_close,
	.cleanup = usbser_tty_cleanup,
	.hangup = usbser_tty_hangup,
	.write = usbser_tty_write,
	.write_room = usbser_tty_write_room,
	.ioctl = usbser_tty_ioctl,
	.chars_in_buffer = usbser_tty_chars_in_buffer,
	.break_ctl = usbser_tty_break_ctl,
	.set_termios = usbser_tty_set_termios,
	.tiocmget = usbser_tty_tiocmget,
	.tiocmset = usbser_tty_tiocmset,
	.get_icount = usbser_get_icount,
};

static int __init usbser_init(void)
{
	int retval;
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0))
	usbser_tty_driver = tty_alloc_driver(USBSER_TTY_MINORS, TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV);
	if (IS_ERR(usbser_tty_driver))
		return PTR_ERR(usbser_tty_driver);
#else
	usbser_tty_driver = alloc_tty_driver(USBSER_TTY_MINORS);
	if (!usbser_tty_driver)
		return -ENOMEM;
#endif
	usbser_tty_driver->driver_name = "usbztekser", usbser_tty_driver->name = "ttyZTEKSER",
	usbser_tty_driver->major = USBSER_TTY_MAJOR, usbser_tty_driver->minor_start = 0,
	usbser_tty_driver->type = TTY_DRIVER_TYPE_SERIAL, usbser_tty_driver->subtype = SERIAL_TYPE_NORMAL,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(5, 15, 0))
	usbser_tty_driver->flags = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV;
#endif
	usbser_tty_driver->init_termios = tty_std_termios;
	usbser_tty_driver->init_termios.c_cflag = B0 | CS8 | CREAD | HUPCL | CLOCAL;
	tty_set_operations(usbser_tty_driver, &usbser_ops);

	retval = tty_register_driver(usbser_tty_driver);
	if (retval) {
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0))
		tty_driver_kref_put(usbser_tty_driver);
#else
		put_tty_driver(usbser_tty_driver);
#endif
		return retval;
	}

	retval = usb_register(&usbser_driver);
	if (retval) {
		tty_unregister_driver(usbser_tty_driver);
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0))
		tty_driver_kref_put(usbser_tty_driver);
#else
		put_tty_driver(usbser_tty_driver);
#endif
		return retval;
	}

	printk(KERN_INFO KBUILD_MODNAME ": " DRIVER_DESC "\n");
	printk(KERN_INFO KBUILD_MODNAME ": " VERSION_DESC "\n");

	return 0;
}

static void __exit usbser_exit(void)
{
	usb_deregister(&usbser_driver);
	tty_unregister_driver(usbser_tty_driver);
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0))
	tty_driver_kref_put(usbser_tty_driver);
#else
	put_tty_driver(usbser_tty_driver);
#endif
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3, 5, 0))
	idr_destroy(&usbser_minors);
#endif
	printk(KERN_INFO KBUILD_MODNAME ": "
					"usbser driver exit.\n");
}

fs_initcall(usbser_init);
module_exit(usbser_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_VERSION(VERSION_DESC);
MODULE_LICENSE("GPL");
MODULE_ALIAS_CHARDEV_MAJOR(USBSER_TTY_MAJOR);
